#!/bin/bash

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
NC='\033[0m' # No Color

# 安装目录
INSTALL_DIR="/opt/webhook-tg-bot"
VENV_DIR="$INSTALL_DIR/venv"
CONFIG_FILE="$INSTALL_DIR/config.json"
SCRIPT_FILE="$INSTALL_DIR/webhook_to_tg_bot.py"
SERVICE_FILE="/etc/systemd/system/webhook-tg-bot.service"

# 检查是否以 root 运行
if [ "$EUID" -ne 0 ]; then
  echo -e "${RED}错误：请以 root 权限运行此脚本（使用 sudo）。${NC}"
  exit 1
fi

# 创建安装目录
echo -e "${GREEN}创建安装目录...${NC}"
mkdir -p "$INSTALL_DIR"
cd "$INSTALL_DIR" || exit 1

# 更新系统并安装依赖
echo -e "${GREEN}更新系统并安装必要工具...${NC}"
apt update && apt install -y python3 python3-pip python3-venv curl systemd

# 创建虚拟环境
echo -e "${GREEN}创建虚拟环境...${NC}"
python3 -m venv "$VENV_DIR"
source "$VENV_DIR/bin/activate"

# 安装 Python 依赖
echo -e "${GREEN}安装 Python 依赖...${NC}"
pip install fastapi uvicorn python-telegram-bot

# 提示输入配置
echo -e "${GREEN}请输入 Telegram Bot Token（回车使用默认值 8300155696:AAGkJMl2CDQfAHwtLxo53LxBIirAMCmLy0w）：${NC}"
read -r TELEGRAM_TOKEN
TELEGRAM_TOKEN=${TELEGRAM_TOKEN:-"8300155696:AAGkJMl2CDQfAHwtLxo53LxBIirAMCmLy0w"}

echo -e "${GREEN}请输入监听端口（回车使用默认值 5000）：${NC}"
read -r PORT
PORT=${PORT:-5000}

# 生成 config.json
echo -e "${GREEN}生成配置文件...${NC}"
cat > "$CONFIG_FILE" << EOL
{
    "telegram_token": "$TELEGRAM_TOKEN",
    "port": $PORT
}
EOL

# 生成 webhook_to_tg_bot.py
echo -e "${GREEN}生成 Python 脚本...${NC}"
cat > "$SCRIPT_FILE" << 'EOL'
import asyncio
import sqlite3
import logging
import json
from fastapi import FastAPI, Request
from contextlib import asynccontextmanager
from telegram import Update
from telegram.ext import Application, CommandHandler, MessageHandler, filters, ContextTypes
from telegram.error import TelegramError

# 配置日志
logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')
logger = logging.getLogger(__name__)

# 读取配置文件
def load_config():
    default_config = {
        "telegram_token": "8300155696:AAGkJMl2CDQfAHwtLxo53LxBIirAMCmLy0w",
        "port": 5000
    }
    try:
        with open('config.json', 'r') as f:
            config = json.load(f)
            port = config.get("port", default_config["port"])
            if not isinstance(port, int):
                logger.warning(f"Port value '{port}' is not an integer, using default {default_config['port']}.")
                port = default_config["port"]
            return {
                "telegram_token": config.get("telegram_token", default_config["telegram_token"]),
                "port": port
            }
    except FileNotFoundError:
        logger.warning("config.json not found, using default values.")
        return default_config
    except (json.JSONDecodeError, ValueError) as e:
        logger.error(f"Invalid config.json format or port value ({e}), using default values.")
        return default_config

# 加载配置
config = load_config()
TELEGRAM_TOKEN = config["telegram_token"]
PORT = config["port"]  # 确保 PORT 全局赋值
logger.info(f"Loaded config - Token: {TELEGRAM_TOKEN[:10]}..., Port: {PORT}")  # 调试日志

# FastAPI app
app = FastAPI()

# SQLite 数据库初始化
conn = sqlite3.connect('bindings.db', check_same_thread=False)
cursor = conn.cursor()
cursor.execute('''
    CREATE TABLE IF NOT EXISTS bindings (
        chat_id INTEGER PRIMARY KEY,
        paused INTEGER DEFAULT 0
    )
''')
conn.commit()

# 全局 Application 实例
application = None

# 发送通知（添加重试逻辑）
async def send_notification_to_bound_chats(message):
    global application
    if application and application.bot:
        cursor.execute('SELECT chat_id FROM bindings WHERE paused = 0')
        rows = cursor.fetchall()
        for row in rows:
            chat_id = row[0]
            retries = 0
            max_retries = 5
            while retries <= max_retries:
                try:
                    await application.bot.send_message(chat_id=chat_id, text=message)
                    logger.info(f"Sent message to chat_id: {chat_id}")
                    break
                except Exception as e:
                    retries += 1
                    logger.error(f"Failed to send to {chat_id} (retry {retries}/{max_retries}): {e}")
                    if retries > max_retries:
                        logger.error(f"Abandoned sending to {chat_id} after {max_retries} retries.")
                        break
                    await asyncio.sleep(1)  # 重试间隔1秒

# Webhook 端点
@app.post('/webhook')
async def webhook(request: Request):
    data = await request.json()
    logger.info(f"Received webhook data: {data}")
    if data and data.get('type') == 'alarm' and 'data' in data and 'message' in data['data']:
        message = data['data']['message']
        await send_notification_to_bound_chats(message)
        return {"status": "ok"}
    return {"status": "error"}, 400

# Telegram 命令处理
async def start(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    chat_id = update.effective_chat.id
    logger.info(f"Received /start from chat_id: {chat_id}")
    await update.message.reply_text('欢迎使用！请使用 /bind 绑定以接收通知。')

async def bind(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    chat_id = update.effective_chat.id
    cursor.execute('SELECT 1 FROM bindings WHERE chat_id = ?', (chat_id,))
    if cursor.fetchone():
        await update.message.reply_text('已经绑定，无需重复操作。')
        return
    cursor.execute('INSERT INTO bindings (chat_id, paused) VALUES (?, 0)', (chat_id,))
    conn.commit()
    logger.info(f"Bound chat_id: {chat_id}")
    await update.message.reply_text('已绑定！现在您将收到所有通知。')

async def unbind(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    chat_id = update.effective_chat.id
    cursor.execute('SELECT 1 FROM bindings WHERE chat_id = ?', (chat_id,))
    if not cursor.fetchone():
        await update.message.reply_text('未绑定，无法执行此操作。请先使用 /bind 绑定。')
        return
    cursor.execute('DELETE FROM bindings WHERE chat_id = ?', (chat_id,))
    conn.commit()
    logger.info(f"Unbound chat_id: {chat_id}")
    await update.message.reply_text('已解除绑定！您将不再收到通知。')

async def pause(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    chat_id = update.effective_chat.id
    cursor.execute('SELECT 1 FROM bindings WHERE chat_id = ?', (chat_id,))
    if not cursor.fetchone():
        await update.message.reply_text('未绑定，无法执行此操作。请先使用 /bind 绑定。')
        return
    cursor.execute('UPDATE bindings SET paused = 1 WHERE chat_id = ?', (chat_id,))
    conn.commit()
    logger.info(f"Paused chat_id: {chat_id}")
    await update.message.reply_text('已暂停推送通知到此会话。')

async def resume(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    chat_id = update.effective_chat.id
    cursor.execute('SELECT 1 FROM bindings WHERE chat_id = ?', (chat_id,))
    if not cursor.fetchone():
        await update.message.reply_text('未绑定，无法执行此操作。请先使用 /bind 绑定。')
        return
    cursor.execute('UPDATE bindings SET paused = 0 WHERE chat_id = ?', (chat_id,))
    conn.commit()
    logger.info(f"Resumed chat_id: {chat_id}")
    await update.message.reply_text('已恢复推送通知到此会话。')

async def unknown(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    logger.info(f"Received unknown message from chat_id: {update.effective_chat.id}")
    await update.message.reply_text('未知指令。请使用 /bind, /unbind, /pause, /resume 或 /start。')

# 手动 polling 循环
async def fetch_updates():
    global application
    offset = 0
    while True:
        try:
            logger.info("Fetching Telegram updates...")
            updates = await application.bot.get_updates(offset=offset, timeout=10)
            for update in updates:
                await application.process_update(update)
                offset = update.update_id + 1
            await asyncio.sleep(0.5)  # 避免过度轮询
        except TelegramError as e:
            logger.error(f"Telegram API error: {e}")
            await asyncio.sleep(1)  # 短暂重试
        except Exception as e:
            logger.error(f"Unexpected error in fetch_updates: {e}")
            await asyncio.sleep(1)

# 启动 Telegram 机器人
async def start_telegram_bot():
    global application
    try:
        logger.info("Initializing Telegram bot...")
        application = Application.builder().token(TELEGRAM_TOKEN).build()
        application.add_handler(CommandHandler("start", start))
        application.add_handler(CommandHandler("bind", bind))
        application.add_handler(CommandHandler("unbind", unbind))
        application.add_handler(CommandHandler("pause", pause))
        application.add_handler(CommandHandler("resume", resume))
        application.add_handler(MessageHandler(filters.TEXT & ~filters.COMMAND, unknown))
        await application.initialize()
        logger.info("Starting Telegram update fetching...")
        asyncio.create_task(fetch_updates())
    except Exception as e:
        logger.error(f"Error initializing Telegram bot: {e}")
        await asyncio.sleep(1)  # 短暂重试

# Lifespan 事件
@asynccontextmanager
async def lifespan(app: FastAPI):
    logger.info("Starting application lifespan...")
    telegram_task = asyncio.create_task(start_telegram_bot())
    yield
    logger.info("Shutting down application...")
    if application:
        await application.shutdown()

# 应用生命周期
app.router.lifespan_context = lifespan

if __name__ == '__main__':
    import uvicorn
    logger.info(f"Starting server on 0.0.0.0:{PORT}")  # 再次确认 PORT
    uvicorn.run(app, host="0.0.0.0", port=PORT)
EOL

# 赋予执行权限
chmod +x "$SCRIPT_FILE"

# 生成 systemd 服务文件
echo -e "${GREEN}生成 systemd 服务文件...${NC}"
cat > "$SERVICE_FILE" << EOL
[Unit]
# Description: Webhook to Telegram Bot Service
# After: Ensures this service starts after the network is available
After=network.target
# Wants: Indicates a dependency on network.target
Wants=network.target

[Service]
# Type: Specifies the service type as simple (runs the process directly)
Type=simple
# User: Runs the service as the root user (change to a non-root user for security)
User=root
# WorkingDirectory: Sets the working directory for the script
WorkingDirectory=$INSTALL_DIR
# ExecStart: Command to start the service (uses virtual environment)
ExecStart=$VENV_DIR/bin/python $SCRIPT_FILE
# Restart: Restarts the service if it crashes
Restart=always
# RestartSec: Wait 5 seconds before restarting
RestartSec=5s
# StartLimitIntervalSec: Disables start limit to allow infinite retries
StartLimitIntervalSec=0
# StandardOutput: Redirects output to systemd journal
StandardOutput=journal
# StandardError: Redirects errors to systemd journal
StandardError=journal
# PrivateTmp: Uses a private tmp directory for security
PrivateTmp=true

[Install]
# WantedBy: Ensures the service starts in multi-user mode
WantedBy=multi-user.target
EOL

# 重新加载 systemd 并启用服务
echo -e "${GREEN}配置 systemd 服务...${NC}"
systemctl daemon-reload
systemctl enable webhook-tg-bot.service
systemctl start webhook-tg-bot.service

# 检查服务状态
echo -e "${GREEN}检查服务状态...${NC}"
systemctl status webhook-tg-bot.service

# 完成提示
echo -e "${GREEN}安装完成！查看日志使用：journalctl -u webhook-tg-bot.service -f${NC}"