#!/bin/bash

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
NC='\033[0m' # No Color

# 安装目录
INSTALL_DIR="/opt/webhook-tg-bot-node"
CONFIG_FILE="$INSTALL_DIR/config.json"
SCRIPT_FILE="$INSTALL_DIR/index.js"
SERVICE_FILE="/etc/systemd/system/webhook-tg-bot-node.service"

# 检查是否以 root 运行
if [ "$EUID" -ne 0 ]; then
  echo -e "${RED}错误：请以 root 权限运行此脚本（使用 sudo）。${NC}"
  exit 1
fi

# 创建安装目录
echo -e "${GREEN}创建安装目录...${NC}"
mkdir -p "$INSTALL_DIR"
cd "$INSTALL_DIR" || exit 1

# 更新系统并安装 Node.js 和依赖
echo -e "${GREEN}更新系统并安装 Node.js...${NC}"
apt update && apt install -y curl gnupg
curl -fsSL https://deb.nodesource.com/setup_18.x | bash -
apt install -y nodejs

# 安装 npm 依赖
echo -e "${GREEN}安装 npm 依赖...${NC}"
npm init -y
npm install express telegraf sqlite3

# 提示输入配置
echo -e "${GREEN}请输入 Telegram Bot Token（回车使用默认值 8300155696:AAGkJMl2CDQfAHwtLxo53LxBIirAMCmLy0w）：${NC}"
read -r TELEGRAM_TOKEN
TELEGRAM_TOKEN=${TELEGRAM_TOKEN:-"8300155696:AAGkJMl2CDQfAHwtLxo53LxBIirAMCmLy0w"}

echo -e "${GREEN}请输入监听端口（回车使用默认值 5000）：${NC}"
read -r PORT
PORT=${PORT:-5000}

# 生成 config.json
echo -e "${GREEN}生成配置文件...${NC}"
cat > "$CONFIG_FILE" << EOL
{
    "telegram_token": "$TELEGRAM_TOKEN",
    "port": $PORT
}
EOL

# 生成 index.js
echo -e "${GREEN}生成 Node.js 脚本...${NC}"
cat > "$SCRIPT_FILE" << 'EOL'
const express = require('express');
const Telegraf = require('telegraf');
const sqlite3 = require('sqlite3').verbose();
const fs = require('fs');
const path = require('path');

const app = express();
const configPath = path.join(__dirname, 'config.json');
const dbPath = path.join(__dirname, 'bindings.db');

// 配置日志
const logger = {
    info: (msg) => console.log(`INFO: ${new Date().toISOString()} - ${msg}`),
    error: (msg) => console.error(`ERROR: ${new Date().toISOString()} - ${msg}`)
};

// 读取配置文件
function loadConfig() {
    const defaultConfig = {
        telegram_token: "8300155696:AAGkJMl2CDQfAHwtLxo53LxBIirAMCmLy0w",
        port: 5000
    };
    try {
        if (fs.existsSync(configPath)) {
            const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));
            const port = parseInt(config.port) || defaultConfig.port;
            if (isNaN(port)) {
                logger.warning(`Port value '${config.port}' is not a valid integer, using default ${defaultConfig.port}.`);
                return defaultConfig;
            }
            return {
                telegram_token: config.telegram_token || defaultConfig.telegram_token,
                port: port
            };
        } else {
            logger.warning("config.json not found, using default values.");
            return defaultConfig;
        }
    } catch (e) {
        logger.error(`Invalid config.json format (${e.message}), using default values.`);
        return defaultConfig;
    }
}

const config = loadConfig();
logger.info(`Loaded config - Token: ${config.telegram_token.substring(0, 10)}..., Port: ${config.port}`);

// 初始化 SQLite 数据库
const db = new sqlite3.Database(dbPath);
db.serialize(() => {
    db.run(`
        CREATE TABLE IF NOT EXISTS bindings (
            chat_id INTEGER PRIMARY KEY,
            paused INTEGER DEFAULT 0
        )
    `);
});

// Telegram Bot
const bot = new Telegraf(config.telegram_token);
let botInitialized = false;

// 发送通知（添加重试逻辑）
async function sendNotificationToBoundChats(message) {
    db.all('SELECT chat_id FROM bindings WHERE paused = 0', [], async (err, rows) => {
        if (err) {
            logger.error(`Database error: ${err.message}`);
            return;
        }
        for (const row of rows) {
            const chatId = row.chat_id;
            let retries = 0;
            const maxRetries = 5;
            while (retries <= maxRetries) {
                try {
                    await bot.telegram.sendMessage(chatId, message);
                    logger.info(`Sent message to chat_id: ${chatId}`);
                    break;
                } catch (e) {
                    retries++;
                    logger.error(`Failed to send to ${chatId} (retry ${retries}/${maxRetries}): ${e.message}`);
                    if (retries > maxRetries) {
                        logger.error(`Abandoned sending to ${chatId} after ${maxRetries} retries.`);
                        break;
                    }
                    await new Promise(resolve => setTimeout(resolve, 1000)); // 重试间隔1秒
                }
            }
        }
    });
}

// Webhook 端点
app.use(express.json());
app.post('/webhook', async (req, res) => {
    logger.info(`Received webhook data: ${JSON.stringify(req.body)}`);
    if (req.body && req.body.type === 'alarm' && req.body.data && req.body.data.message) {
        await sendNotificationToBoundChats(req.body.data.message);
        res.status(200).json({ status: 'ok' });
    } else {
        res.status(400).json({ status: 'error' });
    }
});

// Telegram 命令处理
bot.start((ctx) => {
    logger.info(`Received /start from chat_id: ${ctx.chat.id}`);
    ctx.reply('欢迎使用！请使用 /bind 绑定以接收通知，或使用 /help 查看可用命令。');
});

bot.command('bind', (ctx) => {
    const chatId = ctx.chat.id;
    db.get('SELECT 1 FROM bindings WHERE chat_id = ?', [chatId], (err, row) => {
        if (err) {
            logger.error(`Database error: ${err.message}`);
            return;
        }
        if (row) {
            ctx.reply('已经绑定，无需重复操作。');
        } else {
            db.run('INSERT INTO bindings (chat_id, paused) VALUES (?, 0)', [chatId], (err) => {
                if (err) logger.error(`Database error: ${err.message}`);
                else {
                    logger.info(`Bound chat_id: ${chatId}`);
                    ctx.reply('已绑定！现在您将收到所有通知。');
                }
            });
        }
    });
});

bot.command('unbind', (ctx) => {
    const chatId = ctx.chat.id;
    db.get('SELECT 1 FROM bindings WHERE chat_id = ?', [chatId], (err, row) => {
        if (err) {
            logger.error(`Database error: ${err.message}`);
            return;
        }
        if (!row) {
            ctx.reply('未绑定，无法执行此操作。请先使用 /bind 绑定。');
        } else {
            db.run('DELETE FROM bindings WHERE chat_id = ?', [chatId], (err) => {
                if (err) logger.error(`Database error: ${err.message}`);
                else {
                    logger.info(`Unbound chat_id: ${chatId}`);
                    ctx.reply('已解除绑定！您将不再收到通知。');
                }
            });
        }
    });
});

bot.command('pause', (ctx) => {
    const chatId = ctx.chat.id;
    db.get('SELECT 1 FROM bindings WHERE chat_id = ?', [chatId], (err, row) => {
        if (err) {
            logger.error(`Database error: ${err.message}`);
            return;
        }
        if (!row) {
            ctx.reply('未绑定，无法执行此操作。请先使用 /bind 绑定。');
        } else {
            db.run('UPDATE bindings SET paused = 1 WHERE chat_id = ?', [chatId], (err) => {
                if (err) logger.error(`Database error: ${err.message}`);
                else {
                    logger.info(`Paused chat_id: ${chatId}`);
                    ctx.reply('已暂停推送通知到此会话。');
                }
            });
        }
    });
});

bot.command('resume', (ctx) => {
    const chatId = ctx.chat.id;
    db.get('SELECT 1 FROM bindings WHERE chat_id = ?', [chatId], (err, row) => {
        if (err) {
            logger.error(`Database error: ${err.message}`);
            return;
        }
        if (!row) {
            ctx.reply('未绑定，无法执行此操作。请先使用 /bind 绑定。');
        } else {
            db.run('UPDATE bindings SET paused = 0 WHERE chat_id = ?', [chatId], (err) => {
                if (err) logger.error(`Database error: ${err.message}`);
                else {
                    logger.info(`Resumed chat_id: ${chatId}`);
                    ctx.reply('已恢复推送通知到此会话。');
                }
            });
        }
    });
});

bot.on('text', (ctx) => {
    logger.info(`Received unknown message from chat_id: ${ctx.chat.id}`);
    ctx.reply('未知指令。请使用 /bind, /unbind, /pause, /resume 或 /start。');
});

// 启动服务器
async function startServer() {
    try {
        logger.info(`Starting server on 0.0.0.0:${config.port}`);
        app.listen(config.port, '0.0.0.0', () => {
            logger.info(`Server is running on port ${config.port}`);
        });
    } catch (e) {
        logger.error(`Server startup error: ${e.message}`);
    }
}

// 启动 Telegram Bot
async function startBot() {
    try {
        await bot.launch();
        logger.info("Telegram bot started");
        process.once('SIGINT', () => bot.stop('SIGINT'));
        process.once('SIGTERM', () => bot.stop('SIGTERM'));
    } catch (e) {
        logger.error(`Telegram bot startup error: ${e.message}`);
    }
}

// 启动应用
async function main() {
    await Promise.all([startServer(), startBot()]);
}

main().catch((e) => logger.error(`Main error: ${e.message}`));
EOL

# 赋予执行权限
chmod +x "$SCRIPT_FILE"

# 生成 systemd 服务文件
echo -e "${GREEN}生成 systemd 服务文件...${NC}"
cat > "$SERVICE_FILE" << EOL
[Unit]
# Description: Webhook to Telegram Bot Service (Node.js version)
# After: Ensures this service starts after the network is available
After=network.target
# Wants: Indicates a dependency on network.target
Wants=network.target

[Service]
# Type: Specifies the service type as simple (runs the process directly)
Type=simple
# User: Runs the service as the root user (change to a non-root user for security)
User=root
# WorkingDirectory: Sets the working directory for the script
WorkingDirectory=$INSTALL_DIR
# ExecStart: Command to start the service
ExecStart=/usr/bin/node $SCRIPT_FILE
# Restart: Restarts the service if it crashes
Restart=always
# RestartSec: Wait 5 seconds before restarting
RestartSec=5s
# StartLimitIntervalSec: Disables start limit to allow infinite retries
StartLimitIntervalSec=0
# StandardOutput: Redirects output to systemd journal
StandardOutput=journal
# StandardError: Redirects errors to systemd journal
StandardError=journal
# PrivateTmp: Uses a private tmp directory for security
PrivateTmp=true

[Install]
# WantedBy: Ensures the service starts in multi-user mode
WantedBy=multi-user.target
EOL

# 重新加载 systemd 并启用服务
echo -e "${GREEN}配置 systemd 服务...${NC}"
systemctl daemon-reload
systemctl enable webhook-tg-bot-node.service
systemctl start webhook-tg-bot-node.service

# 检查服务状态
echo -e "${GREEN}检查服务状态...${NC}"
systemctl status webhook-tg-bot-node.service

# 完成提示
echo -e "${GREEN}安装完成！查看日志使用：journalctl -u webhook-tg-bot-node.service -f${NC}"